/** @module obj */

function moduleFunction(config) {
  
  // Initialize $$ with provided config or require debug module
  let $$ = config || require('./debug')({ debug: 'cache', path: module.id, prefix: '[cdev.ncore - fs]' });
  let { nullOrUndefined } = require('./obj');
  let fsRoot = '/agorum/roi/';

  if ($$.environment.name === 'agorum') {
    $$.objects = require('common/objects');
    $$.beans = require('common/beans');
  }

  /**
   * Checks if a file exists
   *
   * @param {string} _file - The file path to check relative to fs root [/agorum/roi/]
   * @returns {boolean} - True if the file exists, false otherwise.
   */
  function fileExists(_file) {
    let file = _file.startsWith('/') ? _file : fsRoot + _file;
    $$.debug('utils/fs - fileExists', file);
    if ($$.environment.name === 'node') {
      let fs = require('fs');
      return fs.existsSync(file);
    } else {
      let obj = $$.objects.tryFind(file);
      return !nullOrUndefined(obj);
    }
  }

  /**
   * Get content of text file
   *
   * @param {string} _file - Filename relative to fs root [/agorum/roi/]
   * @return {string|Buffer} - File content
   */
  function readFile(_file) {
    let file = _file.startsWith('/') ? _file : fsRoot + _file;
    if ($$.environment.name === 'node') {
      let fs = require('fs');
      return fs.readFileSync(file, 'utf8');
    } else {
      let obj = $$.objects.find(file);
      return obj.contentString;
    }
  }

  /**
   * Get files from directory
   *
   * @param {string} _dir - Directory name relative to fs root [/agorum/roi/]
   * @return {string[]} - List of files
   */
  function readDir(_dir) {
    let dir = _dir.startsWith('/') ? _dir : fsRoot + _dir;
    if ($$.environment.name === 'node') {
      let fs = require('fs');
      return (fs.readdirSync(dir) || []).filter(function(name) { 
        return !name.startsWith('.');
      });
    } else {
      let obj = $$.objects.find(dir);
      return (obj.items() || [])
        .filter(function(item) { 
          return !item.name.startsWith('.');
        })
        .map(function(item) { 
          return item.name;
        });
    }
  }

  /**
   * Creates a file object with the given data.
   * @param {Object} data - The data to create the file object with.
   * @returns {Object} The created file object.
   */
  function createFile(data) {
    let file = null,
      acl,
      content = null,
      update = true;

    if (data.acl) {
      acl = data.acl;
      delete data.acl;
    }
    if (data.content !== undefined) {
      content = data.content;
      delete data.content;
    }
    if (data.update !== undefined) {
      update = data.update;
      delete data.update;
    }

    if ($$.environment.name === 'agorum') {
      if (update) {
        file = $$.objects.tryCreate('file', data);
      } else {
        file = $$.objects.create('file', data);
      }
      if (file) $$.beans.set(file, { content: content });
      if (file && acl) file.acl = acl;
    } else {
      const fs = require('fs');
      const filePath = fsRoot + (data.target || '') + '/' + data.name;
      const updated = fs.existsSync(filePath);
      fs.writeFileSync(filePath, content, 'utf8');
      file = { path: filePath, updated: updated };
    }
    return file;
  }

  /**
   * Writes content to a file.
   *
   * @param {string} filePath - The path to the file, relative to the fs root [/agorum/roi/] or absolute.
   * @param {string} content - The content to write to the file.
   * @returns {Object|null} - The file object or null if an error occurred.
   */
  function writeFile(filePath, content) {
    if ($$.environment.name === 'agorum') {
      let file = $$.objects.tryCreate('file', {
        name: filePath.split('/').pop(),
        target: $$.objects.find(filePath.split('/').slice(0, -1).join('/')),
      });
      if (file) {
        $$.beans.set(file, { content: content });
        return file;
      } else {
        return null;
      }
    } else {
      const fs = require('fs');
      const absoluteFilePath = filePath.startsWith('/') ? filePath : fsRoot + filePath;
      try {
        fs.writeFileSync(absoluteFilePath, content, 'utf8');
        return { path: absoluteFilePath };
      } catch (error) {
        $$.debug('cdev.ncore.fs.writeFile - error writing file:', error);
        return null;
      }
    }
  }

  /**
   * Deletes a file.
   *
   * @param {string} filePath - The path to the file, relative to the fs root [/agorum/roi/] or absolute.
   * @returns {boolean} - True if the file was deleted successfully, false otherwise.
   */
  function deleteFile(filePath) {
    if ($$.environment.name === 'agorum') {
      let file = $$.objects.tryFind(filePath);
      if (file) {
        file.delete();
        return true;
      } else {
        return false;
      }
    } else {
      const fs = require('fs');
      const absoluteFilePath = filePath.startsWith('/') ? filePath : fsRoot + filePath;
      try {
        fs.unlinkSync(absoluteFilePath);
        return true;
      } catch (error) {
        $$.debug('cdev.ncore.fs.deleteFile - error deleting file:', error);
        return false;
      }
    }
  }

  // Return the module functions
  return {
    fileExists: fileExists,
    readFile: readFile,
    readDir: readDir,
    createFile: createFile,
    writeFile: writeFile,
    deleteFile: deleteFile
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
